﻿/*
	Name:			XenoN Core Audio Component Header
	Version:		1.4
	Update:			2011-02-24
	Copyright:		Copyright © 2007-2011 XenoN Core by PsichiX. All rights reserved.
	Author:			PsichiX
	Website:		http://www.xenon.psichix.com
	Description:	XenoN Core Framework Audio Component Header
*/

/*
	== EN ==
	This file is part of Xenon Core Framework.
	You may distribute it further, but you can not modify it.
	Please do not use in modified form.
	The principles of XenoN Core License available in the file LICENSE_CORE_EN.TXT or visit: http://www.xenon.psichix.com.

	== PL ==
	Ten plik jest czescia XenoN Core Framework.
	Mozesz go rozpowszechniac dalej, jednak nie mozesz go modyfikowac.
	Nalezy uzytkowac w nie zmodyfikowanej formie.
	Nalezy przestrzegac zasad Licencji XenoN Core dostepnej w pliku LICENSE_CORE_PL.TXT oraz na stronie: http://www.xenon.psichix.com.
*/

#ifndef XE_COMPONENT_AUDIO_H
#define XE_COMPONENT_AUDIO_H

#include "../System/XenonCoreFramework.h"

#pragma comment(lib,"Components/Audiere/audiere.lib")
#include "Audiere/audiere.h"

using namespace audiere;

//! Przestrzen nazw XenoN Core
namespace XeCore
{
//! Przestrzen nazw komponentow
namespace Com
{

struct ID3V1_TAG;
class C_SOUND_BUFFER;
class C_MUSIC_BUFFER;
class C_AUDIO;

#ifdef XE_COMPILE_ECHO

//! Struktura tagu ID3v1.
struct ID3V1_TAG
{
	char	Title[30];
	char	Artist[30];
	char	Album[30];
	char	Year[4];
	char	Comment[28];
	char	Track[2];
	char	Genre[1];
};

//! Klasa bufora dzwieku.
class C_SOUND_BUFFER
{
	friend class C_AUDIO;
private:
	SoundEffectPtr	Sound;
	FilePtr			File;
	XE_BUFFER		Buffer;
public:
					~C_SOUND_BUFFER();
};

C_SOUND_BUFFER::~C_SOUND_BUFFER()
{
	Buffer.Free();
}

//! Klasa bufora utworu.
class C_MUSIC_BUFFER
{
	friend class C_AUDIO;
private:
	OutputStreamPtr	Music;
	FilePtr			File;
	XE_BUFFER		Buffer;
public:
					~C_MUSIC_BUFFER();
};

C_MUSIC_BUFFER::~C_MUSIC_BUFFER()
{
	Buffer.Free();
}

//! Definicja elementu dzwieku.
typedef	XE_ELEMENT_POINTER<C_SOUND_BUFFER>	C_ELM_SOUND;
//! Definicja elementu utworu.
typedef	XE_ELEMENT_POINTER<C_MUSIC_BUFFER>	C_ELM_MUSIC;

//! Klasa audio
class C_AUDIO
{
private:
	AudioDevicePtr					Device;
	XE_ELEMENTS<C_SOUND_BUFFER>		Sounds;
	XE_ELEMENTS<C_MUSIC_BUFFER>		Musics;
public:
									/*! Konstruktor obiektu. */
									C_AUDIO();
									/*! Destruktor obiektu. */
									~C_AUDIO();
									/*! Dodaje nowy dzwiek.
									\param fname Cstring z nazwa pliku.
									\return Element dzwieku.
									*/
	C_ELM_SOUND						SoundAdd(char* fname);
									/*! Dodaje nowy dzwiek z buforu.
									\param data Wskaznik na dane dzwieku.
									\param size Rozmiar danych w bajtach.
									\return Element dzwieku.
									*/
	C_ELM_SOUND						SoundAddFromMemory(void* data,unsigned int size);
									/*! Czysci kontener dzwiekow.
									\return true jesli kontener posiadal elementy.
									*/
	bool							SoundClean();
									/*! Zwraca iterator na pierwszy iterator dzwieku.
									\return Pierwszy element dzwieku z kontenera.
									*/
	C_ELM_SOUND						SoundFirst();
									/*! Zwalnia dany dzwiek.
									\param sound Element dzwieku.
									\return true jesli operacja powiodla sie.
									*/
	bool							SoundFree(C_ELM_SOUND sound);
									/*! Odtwarza dany dzwiek.
									\param sound Element dzwieku.
									\return true jesli operacja powiodla sie.
									*/
	bool							SoundPlay(C_ELM_SOUND sound);
									/*! Zatrzymuje dany dzwiek.
									\param sound Element dzwieku.
									\return true jesli operacja powiodla sie.
									*/
	bool							SoundStop(C_ELM_SOUND sound);
									/*! Ustala poziom glosnosci danego dzwieku.
									\param sound Element dzwieku.
									\param volume Poziom glosnosci (zakres: <0;1>).
									\return true jesli operacja powiodla sie.
									*/
	bool							SoundVolume(C_ELM_SOUND sound,float volume);
									/*! Zwraca poziom glosnosci dzwieku.
									\param sound Element dzwieku.
									\return Poziom glosnosci.
									*/
	float							SoundVolumeGet(C_ELM_SOUND sound);
									/*! Ustala balans glosnosci dzwieku.
									\param sound Element dzwieku.
									\param pan Balans dzwieku (zakres: <-1;1>).
									\return true jesli operacja powiodla sie.
									*/
	bool							SoundPan(C_ELM_SOUND sound,float pan);
									/*! Zwraca balans glosnosci dzwieku.
									\param sound Element dzwieku.
									\return Balans glosnosci.
									*/
	float							SoundPanGet(C_ELM_SOUND sound);
									/*! Ustala tonacje dzwieku.
									\param sound Element dzwieku.
									\param pitch Tonacja dzwieku (zakres: <0.5;2>).
									\return true jesli operacja powiodla sie.
									*/
	bool							SoundPitch(C_ELM_SOUND sound,float pitch);
									/*! Zwraca tonacje dzwieku.
									\param sound Element dzwieku.
									\return Tonacja.
									*/
	float							SoundPitchGet(C_ELM_SOUND sound);
									/*! Dodaje nowy utwor.
									\param fname Cstring z nazwa pliku.
									\return Element utworu.
									*/
	C_ELM_MUSIC						MusicAdd(char* fname);
									/*! Dodaje nowy utwor z buforu.
									\param data Wskaznik na dane utworu.
									\param size Rozmiar danych w bajtach.
									\return Element utworu.
									*/
	C_ELM_MUSIC						MusicAddFromMemory(void* data,unsigned int size);
									/*! Czysci kontener utworow.
									\return true jesli kontener posiadal elementy.
									*/
	bool							MusicClean();
									/*! Zwraca iterator na pierwszy utwor.
									\return Pierwszy element utworu z kontenera.
									*/
	C_ELM_MUSIC						MusicFirst();
									/*! Zwalnia dany utwor.
									\param music Element utworu.
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicFree(C_ELM_MUSIC music);
									/*! Odtwarza dany utwor.
									\param music Element utworu.
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicPlay(C_ELM_MUSIC music);
									/*! Zatrzymuje dany utwor.
									\param music Element utworu.
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicStop(C_ELM_MUSIC music);
									/*! Ustala poziom glosnosci danego utworu.
									\param music Element utworu.
									\param volume Poziom glosnosci (zakres: <0;1>).
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicVolume(C_ELM_MUSIC music,float volume);
									/*! Zwraca poziom glosnosci utworu.
									\param music Element utworu.
									\return Poziom glosnosci.
									*/
	float							MusicVolumeGet(C_ELM_MUSIC music);
									/*! Ustala balans glosnosci utworu.
									\param music Element utworu.
									\param pan Balans dzwieku (zakres: <-1;1>).
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicPan(C_ELM_MUSIC music,float pan);
									/*! Zwraca balans glosnosci utworu.
									\param music Element utworu.
									\return Balans glosnosci.
									*/
	float							MusicPanGet(C_ELM_MUSIC music);
									/*! Ustala tonacje utworu.
									\param music Element utworu.
									\param pitch Tonacja utworu (zakres: <0.5;2>).
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicPitch(C_ELM_MUSIC music,float pitch);
									/*! Zwraca tonacje utworu.
									\param music Element utworu.
									\return Tonacja.
									*/
	float							MusicPitchGet(C_ELM_MUSIC music);
									/*! Resetuje dany utwor.
									\param music Element utworu.
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicReset(C_ELM_MUSIC music);
									/*! Sprawdza czy dany utwor jest odtwarzany.
									\param music Element utworu.
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicIsPlaying(C_ELM_MUSIC music);
									/*! Ustala pozycje odtwarzania.
									\param music Element utworu.
									\param pos Pozycja (numer ramki).
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicSeek(C_ELM_MUSIC music,int pos);
									/*! Zwraca pozycje w utworze.
									\param music Element utworu.
									\return Pozycja (numer ramki).
									*/
	int								MusicPosition(C_ELM_MUSIC music);
									/*! Zwraca ilosc ramek utworu.
									\param music Element utworu.
									\return Ilosc ramek, lub 0 jesli utwor nie ma mozliwosci ustalania pozycji.
									*/
	int								MusicLength(C_ELM_MUSIC music);
									/*! Ustala czy utwor ma sie powtarzac.
									\param music Element utworu.
									\param repeat Powtarzanie.
									\return true jesli operacja powiodla sie.
									*/
	bool							MusicRepeat(C_ELM_MUSIC music,bool repeat);
									/*! Pobiera tag ID3v1.1.
									\param filename Nazwa pliku.
									\param buffer Wskaznik na bufor tagu.
									*/
	static bool						GetID3v1(char* filename,ID3V1_TAG* buffer);
};

#ifdef XE_CANIMP

C_AUDIO::C_AUDIO()
{
	Device=OpenDevice();
}

C_AUDIO::~C_AUDIO()
{
	Sounds.Clear();
	Musics.Clear();
}

C_ELM_SOUND C_AUDIO::SoundAdd(char* fname)
{
	C_ELM_SOUND elm;
	elm=Sounds.AddPointer(C_SOUND_BUFFER());
	if(elm.IsEmpty())return(C_ELM_SOUND());
	elm->Sound=OpenSoundEffect(Device,fname,MULTIPLE);
	if(!elm->Sound)
	{
		Sounds.ErasePointer(elm);
		elm.Unref();
	}
	return(elm);
}

C_ELM_SOUND C_AUDIO::SoundAddFromMemory(void* data,unsigned int size)
{
	C_ELM_SOUND elm;
	elm=Sounds.AddPointer(C_SOUND_BUFFER());
	if(elm.IsEmpty())return(C_ELM_SOUND());
	elm->Buffer.Reserve(size);
	elm->Buffer.WriteBytes(data,size);
	elm->File=CreateMemoryFile(elm->Buffer.Get(),elm->Buffer.Size());
	if(elm->File==NULL)
	{
		Sounds.ErasePointer(elm);
		return(C_ELM_SOUND());
	}
	elm->Sound=OpenSoundEffect(Device,elm->File,MULTIPLE);
	if(!elm->Sound)
	{
		Sounds.ErasePointer(elm);
		return(C_ELM_SOUND());
	}
	return(elm);
}

bool C_AUDIO::SoundClean()
{
	if(Sounds.Size()==0)return(false);
	Sounds.Clear();
	return(true);
}

C_ELM_SOUND C_AUDIO::SoundFirst()
{
	return(Sounds.FirstPointer());
}

bool C_AUDIO::SoundFree(C_ELM_SOUND sound)
{
	if(sound.IsEmpty())return(false);
	Sounds.ErasePointer(sound);
	return(true);
}

bool C_AUDIO::SoundPlay(C_ELM_SOUND sound)
{
	if(sound.IsEmpty())return(false);
	sound->Sound->play();
	return(true);
}

bool C_AUDIO::SoundStop(C_ELM_SOUND sound)
{
	if(sound.IsEmpty())return(false);
	sound->Sound->stop();
	return(true);
}

bool C_AUDIO::SoundVolume(C_ELM_SOUND sound,float volume)
{
	if(sound.IsEmpty())return(false);
	sound->Sound->setVolume(volume);
	return(true);
}

float C_AUDIO::SoundVolumeGet(C_ELM_SOUND sound)
{
	if(sound.IsEmpty())return(0);
	return(sound->Sound->getVolume());
}

bool C_AUDIO::SoundPan(C_ELM_SOUND sound,float pan)
{
	if(sound.IsEmpty())return(false);
	sound->Sound->setPan(pan);
	return(true);
}

float C_AUDIO::SoundPanGet(C_ELM_SOUND sound)
{
	if(sound.IsEmpty())return(0);
	return(sound->Sound->getPan());
}

bool C_AUDIO::SoundPitch(C_ELM_SOUND sound,float pitch)
{
	if(sound.IsEmpty())return(false);
	sound->Sound->setPitchShift(pitch);
	return(true);
}

float C_AUDIO::SoundPitchGet(C_ELM_SOUND sound)
{
	if(sound.IsEmpty())return(0);
	return(sound->Sound->getPitchShift());
}

C_ELM_MUSIC C_AUDIO::MusicAdd(char* fname)
{
	C_ELM_MUSIC elm;
	elm=Musics.AddPointer(C_MUSIC_BUFFER());
	if(elm.IsEmpty())return(C_ELM_MUSIC());
	elm->Music=OpenSound(Device,fname,true);
	if(!elm->Music)
	{
		Musics.ErasePointer(elm);
		elm.Unref();
	}
	return(elm);
}

C_ELM_MUSIC C_AUDIO::MusicAddFromMemory(void* data,unsigned int size)
{
	C_ELM_MUSIC elm;
	elm=Musics.AddPointer(C_MUSIC_BUFFER());
	if(elm.IsEmpty())return(C_ELM_MUSIC());
	elm->Buffer.Reserve(size);
	elm->Buffer.WriteBytes(data,size);
	elm->File=CreateMemoryFile(elm->Buffer.Get(),elm->Buffer.Size());
	if(elm->File==NULL)
	{
		Musics.ErasePointer(elm);
		return(C_ELM_MUSIC());
	}
	elm->Music=OpenSound(Device,elm->File,true);
	if(!elm->Music)
	{
		Musics.ErasePointer(elm);
		return(C_ELM_MUSIC());
	}
	return(elm);
}

bool C_AUDIO::MusicClean()
{
	if(Musics.Size()==0)return(false);
	Musics.Clear();
	return(true);
}

C_ELM_MUSIC C_AUDIO::MusicFirst()
{
	return(Musics.FirstPointer());
}

bool C_AUDIO::MusicFree(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	Musics.ErasePointer(music);
	return(true);
}

bool C_AUDIO::MusicPlay(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	music->Music->play();
	return(true);
}

bool C_AUDIO::MusicStop(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	music->Music->stop();
	return(true);
}

bool C_AUDIO::MusicVolume(C_ELM_MUSIC music,float volume)
{
	if(music.IsEmpty())return(false);
	music->Music->setVolume(volume);
	return(true);
}

float C_AUDIO::MusicVolumeGet(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	return(music->Music->getVolume());
}

bool C_AUDIO::MusicPan(C_ELM_MUSIC music,float pan)
{
	if(music.IsEmpty())return(false);
	music->Music->setPan(pan);
	return(true);
}

float C_AUDIO::MusicPanGet(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	return(music->Music->getPan());
}

bool C_AUDIO::MusicPitch(C_ELM_MUSIC music,float pitch)
{
	if(music.IsEmpty())return(false);
	music->Music->setPitchShift(pitch);
	return(true);
}

float C_AUDIO::MusicPitchGet(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	return(music->Music->getPitchShift());
}

bool C_AUDIO::MusicReset(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	music->Music->reset();
	return(true);
}

bool C_AUDIO::MusicIsPlaying(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(false);
	return(music->Music->isPlaying());
}

bool C_AUDIO::MusicSeek(C_ELM_MUSIC music,int pos)
{
	if(music.IsEmpty())return(false);
	music->Music->setPosition(pos);
	return(true);
}

int C_AUDIO::MusicPosition(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(0);
	return(music->Music->getPosition());
}

int C_AUDIO::MusicLength(C_ELM_MUSIC music)
{
	if(music.IsEmpty())return(0);
	return(music->Music->getLength());
}

bool C_AUDIO::MusicRepeat(C_ELM_MUSIC music, bool repeat)
{
	if(music.IsEmpty())return(false);
	music->Music->setRepeat(repeat);
	return(true);
}

bool C_AUDIO::GetID3v1(char* filename,ID3V1_TAG* buffer)
{
	FILE* fp=NULL;
	fopen_s(&fp,filename,"rb");
	if(!fp)return(false);
	char hastag[4];
	memset(hastag,0,4);
	fseek(fp,-128, SEEK_END);
	fread(hastag,1,3,fp);
	if(strcmp(hastag,"TAG"))
	{
		fclose(fp);
		return(false);
	}
	fread((void*)buffer,1,sizeof(ID3V1_TAG),fp);
	fclose(fp);
	return(true);
}

#endif /* XE_CANIMP */

#endif /* XE_COMPILE_ECHO */

} /* namespace: Com */
} /* namespace: XeCore */

#endif /* XE_COMPONENT_AUDIO_H */
